! Opens <profileFileName> in <OutputDirectory>
! Writes section number, resonance frequency, maximum position and maximum velocity
! of each plotted section to the file and closes it.

! If <storeExtraInfo>, a header row will be written, indicating
! what values will be written in each colum, that is:
! "section", "resonance freq. (Hz)", "max. position (m)", "max. velocity (m/s)"
! and the parameters will be written at the end of the file using a call to Parameters.

! If <storeBinary>, the values will be written as real(4)'s in a binary file. No extra info
! will be added. To process the binary data in Matlab use:
! fid = fopen ( '    profileFileName    ' ) 
! x = fread(fid, [4, inf], 'real*4')
! section = x(:, 1)
! resonaceFrequency = x(:, 2)
! maxPosition = x(:, 3)
! maxVelocity = x(:, 4)

SUBROUTINE WriteProfile
	USE Declare
	USE GraphsModule
	USE FilesModule
	IMPLICIT NONE
	REAL f_resonance(n)
	INTEGER ii, j, ios, itplot0, itplotN
	IF (storeProfile) THEN

		OPEN (FH_profile, FILE = TRIM(outputDirectory) // TRIM(profileFileName), FORM = writeFormat, IOSTAT = ios)

        IF (storeExtraInfo) THEN
			WRITE (FH_profile, '(A7, A23, A23, A23, A23, A23)', IOSTAT = ios) "section", "resonance freq. (Hz)", &
&			    "max. position (m)", "position phase (rad/pi)", "max. velocity (m/s)", "velocity phase (rad/pi)"
			WRITE (FH_profile, *, IOSTAT = ios)
		ENDIF

		f_resonance = SQRT(sdivm(1:n)) / (2. * pi)
		itplotN = t_plotpoints-1
		itplot0 = int(t_plotpoints * 3/4)

		IF (storeBinary) THEN
			WRITE (FH_profile, IOSTAT = ios) REAL(0), f_resonance(1), &
&				 MAXVAL(positionStapes(itplot0:itplotN)), StapesPositionPhase, MAXVAL(velocityStapes(itplot0:itplotN)), StapesVelocityPhase
		ELSE
				WRITE (FH_profile, '(I7, A23, ES23.7, ES23.7, ES23.7, ES23.7)', IOSTAT = ios) 0, " stapes ", &
&				 MAXVAL(positionStapes(itplot0:itplotN)), StapesPositionPhase, MAXVAL(velocityStapes(itplot0:itplotN)), StapesVelocityPhase
		ENDIF

		DO i = 1, n, plotEvery_n_points
			ii = (i-1) / plotEvery_n_points
			IF (storeBinary) THEN
				WRITE (FH_profile, IOSTAT = ios) REAL(i), f_resonance(i), &
&				 MAXVAL(positionMatrix(ii, itplot0:itplotN)), PositionPhase(ii), MAXVAL(velocityMatrix(ii, itplot0:itplotN)), VelocityPhase(ii)
			ELSE
				WRITE (FH_profile, '(I7, ES23.7, ES23.7, ES23.7, ES23.7, ES23.7)', IOSTAT = ios) i, f_resonance(i), &
&				 MAXVAL(positionMatrix(ii, itplot0:itplotN)), PositionPhase(ii), MAXVAL(velocityMatrix(ii, itplot0:itplotN)), VelocityPhase(ii)
			ENDIF
		END DO

		IF (storeExtraInfo) THEN
			WRITE (FH_profile, *, IOSTAT = ios)
			CALL Parameters (WRITES, FH_profile)
			IF (.NOT. ParameterFileSucceeded) CALL WriteError(profileFileName)
		ENDIF

		CLOSE (FH_profile)
	
	ENDIF

END SUBROUTINE WriteProfile